subroutine neighbours(npart, box, coords, rcutoff, selectedneighbors, Nb)

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!     Fortran implementation of the SANN algorithm                        !
!                                                                         !
!     van Meel, Filion, Valeriani and Frenkel November (2011)             !
!                                                                         !
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!     
!     Adapted by Sam Niblett for use with pele, Aug 2015

!     Some notes (added by S.N.):

!     To compile for use with python:
!     f2py -c --fcompiler='gnu95' -m sann sann.f90 only: neighbours :

!     The maximum number of neighbours which can be detected is hard-coded, to 100.
!     This should be more than large enough for all anticipated applications, and many
!     users may be able to reduce this number safely.

!     For portability and simplicity of compilation, I have included a Fortran
!     implementation of quicksort which is called from the main subroutine "neighbours",
!     rather than using an external library routine.


!     declare all variables used in the subroutine
implicit none

!     npart = total number of particles in the system
integer, intent(in) :: npart
!     simulation box dimensions
double precision, intent(in), dimension(3) :: box
!     cartesian coords for all particles in the system
double precision, intent(in), dimension(npart*3) :: coords
!     cutoff distance to identify all potential neighbours
double precision, intent(in) :: rcutoff

!     selectedneighbors = list of selected neighbours
integer, intent(out) :: selectedneighbors(npart,100)
!     Nb = final number of neighbours of particle i
integer, intent(out) :: Nb(npart)

!     m = tentative number of neighbours
integer i,j,k,m
!     countneighbors = number of neighbours of particle i
integer countneighbors(npart)
!     sortneighbor = sorted neighbours 
integer sortneighbor(npart,100)
!     neighbor = list of neighbours of particles i
integer neighbor(npart,100)
!     distance = list of distances between each 
!     neighbour of particle i and particle i 
double precision  distance(npart,100)
!     distancesorted = sorted distances
double precision distancesorted(npart,100)
!     R(m) as in Eq.3 in the manuscript
double precision rm,rm1
!     x,y,z component of every particle i
double precision x(npart),y(npart),z(npart)
!     distance between particle i and particle j
double precision dx, dy, dz, dr
      
external :: sort

countneighbors(:)=0
selectedneighbors(:,:)=0
Nb(:)=0
sortneighbor(:,:)=0
neighbor(:,:)=0
distance(:,:)=0
distancesorted(:,:)=0

!     Preparation: fill the position arrays from the coords file
do i=1,npart
    x(i) = coords(3*i-2)
    y(i) = coords(3*i-1)
    z(i) = coords(3*i)
enddo
!     Step 1:
!     first we identify the particles within a cutoff radius rcutoff
do i=1,npart
!     loop over all particles different from i
   do j=1,npart
       if (j.ne.i)then
!         compute x,y,z component of the distance between particle i and j
          dx = x(j) - x(i)
          dy = y(j) - y(i)
          dz = z(j) - z(i)

!         apply periodic boundary conditions
          dx=dx-nint(dx/box(1))*box(1)
          dy=dy-nint(dy/box(2))*box(2)
          dz=dz-nint(dz/box(3))*box(3)

!          compute distance dr  between particle i and j
          dr =  sqrt(dx*dx+dy*dy+dz*dz)

!         identify neighbours that are within a cutoff (rcutoff)
          if(dr.lt.rcutoff)then
       !     j is a neighbour of i
             countneighbors(i) = countneighbors(i) + 1
             if (countneighbors(i) .gt. 100) write(*,*) "Too many neighbours for atom ", i

!            build a list of neighbours
             neighbor(i,countneighbors(i))= j
!            create a list containing the distance between i and j
             distance(i,countneighbors(i))=dr
          endif
       endif
   enddo
enddo

!     Step 2:
!     for every particle i sort all (countneighbors) 
!     neighbours (neighbor) according to their 
!     distances (distance) and create  a new list of 
!     particle i's (sortneighbor)
!     and a new sorted list of distances (distancesorted)
do i=1,npart
    call sort(npart,i,countneighbors,distance,neighbor, &
     &        sortneighbor,distancesorted)
enddo

do i=1,npart
!     Step 3: 
!     start with 3 neighbours
    m = 3
!     Step 4: 
!     compute R(m) as in Eq.3 of the van Meel paper
    rm = 0
    do k=1,m
       rm = rm + distancesorted(i,k)
    enddo
    rm = rm/(m-2)
!     compute r(m+1)
    do j = 1,countneighbors(i)
       rm1 = 0
       do k=1,m
          rm1 = rm1 + distancesorted(i,k)
       enddo
       rm1 = rm1/(m-2)
!     Step 5:  
!     if rm > rm1...
       if(rm.ge.rm1)then
          rm = rm1
!     ...increase m
          m = m+1
       else
!     Step 6:
!     if rm < rm1, m is the final number of neighbours (algorithm has converged)
          exit
       endif
    enddo
!     the final number of neighbours is m = Nb(i) 
!     and the neighbours are  selectedneighbors
    Nb(i) = m
    do j=1,Nb(i)
       selectedneighbors(i,j) = sortneighbor(i,j)
    enddo
enddo

if (any(selectedneighbors(:,:)>npart)) THEN
   write(*,*) "Warning: some neighbours are greater than npart"
   stop
endif

return
end


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!      

subroutine sort(npart,i,countneighbors,distance,neighbor, &
     &        sortneighbor,distancesorted)
!     sort all (countneighbors) neighbours (neighbor) of
!     particle i according to their
!     distances (distance) and create  a new list of
!     particle i's (sortneighbor)
!     and a new sorted list of distances (distancesorted)

! A recursive quicksort, QsortC, is used (see below for citation)

  implicit none
  integer, intent(in) :: npart
  integer, intent(in) :: i
  integer, intent(in) :: countneighbors(npart)

  integer, intent(in) :: neighbor(npart,100)
  double precision, intent(in) ::  distance(npart,100)
  integer, intent(out) :: sortneighbor(npart,100)
  double precision, intent(out) :: distancesorted(npart,100)

  integer :: work1(countneighbors(i))
  double precision :: work2(countneighbors(i))

  interface
    subroutine QsortC(A, compare)
      integer, intent(in out), dimension(:) :: A
      double precision, intent(in out), dimension(:) :: compare
    end subroutine QsortC
  end interface

  work1(:) = neighbor(i,:countneighbors(i))
  work2(:) = distance(i,:countneighbors(i))
  call QsortC(work1, work2)
  sortneighbor(i,:countneighbors(i)) = work1(:)
  distancesorted(i,:countneighbors(i)) = work2(:)

  return

end subroutine sort

recursive subroutine QsortC(A, compare)

! Recursive Fortran 95 quicksort routine
! sorts real numbers into ascending numerical order
! Author: Juli Rew, SCD Consulting (juliana@ucar.edu), 9/03
! Based on algorithm from Cormen et al., Introduction to Algorithms,
! 1997 printing

! Made F conformant by Walt Brainerd

! Obtained from www.fortran.com

  implicit none
  integer, intent(in out), dimension(:) :: A
!f2py intent(in,out) :: A
  double precision, intent(in out), dimension(:) :: compare
  integer :: iq

  interface
    subroutine Partition(A, compare, marker)
      integer, intent(in out), dimension(:) :: A
      double precision, intent(in out), dimension(:) :: compare
      integer, intent(out) :: marker
    end subroutine Partition
  end interface

  if(size(A) > 1) then
     call Partition(A, compare, iq)
     call QsortC(A(:iq-1),compare(:iq-1))
     call QsortC(A(iq:),compare(iq:))
  endif
end subroutine !QsortC

subroutine Partition(A, compare, marker)
  implicit none
  integer, intent(in out), dimension(:) :: A
!f2py intent(in,out) :: A
  double precision, intent(in out), dimension(:) :: compare
  integer, intent(out) :: marker
  integer :: i, j, temp
  double precision :: temp2
  double precision :: x      ! pivot point

  x = compare(1)
  i= 0
  j= size(A) + 1

  do
     j = j-1
     do
        if (compare(j) <= x) exit
        j = j-1
     end do
     i = i+1
     do
        if (compare(i) >= x) exit
        i = i+1
     end do
     if (i < j) then
        ! exchange A(i) and A(j)
        temp = A(i)
        temp2 = compare(i)
        A(i) = A(j)
        compare(i) = compare(j)
        A(j) = temp
        compare(j) = temp2
     elseif (i == j) then
        marker = i+1
        return
     else
        marker = i
        return
     endif
  end do

end subroutine !Partition

!end module
