import numpy as np
from pele.utils.rotations import q2aa

# A library of classes to read in lists of configurations from plaintext input files.
# These classes assume that the configurations are simply concatenated, without any indication of the number of atoms
# per configuration and without atom symbols. A class to read xyz-format files will follow in due course.

# For very long lists of configurations (e.g. long MD trajectories), the classes can read in a list of files one after
# the other.

# Note: this file was written for a project studying a rigid-body model for OTP, so the "vanilla" function ReadConfigurations
# is appropriate only for rigid-body coordinates written in the angle-axis formalism. It requires a copy of pele 
# (https://github.com/pele-python/pele) on your PYTHONPATH.

class FileEmptyError(IOError):
    pass

def _load_matrix(fin, nrow, ncol):
    """ Load a matrix of floats from a file "fin" into a numpy array of dimensions (nrow,ncol) """
    x = np.zeros([nrow, ncol])
    # Each row of the matrix should be on a separate line in the file
    for i in xrange(nrow):
        line = fin.readline()
        if line == "":
            if i == 0:
                raise FileEmptyError()
            raise IOError("Not enough lines for the matrix. Number of lines found: "+str(i))
        sline = line.split()
        if len(sline) != ncol:
            raise IOError("Wrong number of columns, line: "+line)
        # Convert the strings into floats
        x[i,:] = map(float, sline)
    
    return x
   
class ReadConfigurations(object):
    """ Read a file containing the positions and orientations of a system of rigid bodies, using angle-axis coordinates
        to describe the orientations.
        Multiple configurations may be contained within the file. Each configuration is composed of centre-of-mass
        coordinates followed by angle-axis coordinates. 1-column or 3-column formats are both supported.
        The coordinates may be accessed by iterating over self.configuration_iterator(), which reads in each
        configuration as it is required - this is particularly efficient when you only need to read part of the file.
        The returned coordinates will be given as np arrays with the same shape and structure as the original file.
        
        Parameters
        ----------
        coords_fname: string
            Name of the file containing centre-of-mass coordinates for the rigid bodies, followed by angle-axis
        nmol: int
            The number of rigid bodies in the system
        ncol: int, default 1
            The number of columns used for the coordinate matrices in the input file
    """    
    def __init__(self, coords_fname, nmol, ncol=1):    
        self.coords_file = open(coords_fname, "r")
        self.nmol = nmol
        self.configuration_count = 0
        self.ncol = ncol
        
    def configuration_iterator(self):
        """ Return the next configuration from the file every time this function is called."""
        while self.coords_file:
            try:
                x = self.load_configuration()
            except FileEmptyError:
                # this means we have reached the end of the file normally
                break
            self.configuration_count += 1
            yield x
            
    def load_configuration(self):
        """ Read the next configuration in from the file """
        # ndof is the number of degrees of freedom in the system
        ndof = 6*self.nmol
        # This should never fail if ncol is set to 1 or 3.
        assert ndof % self.ncol == 0
        nrow = ndof / self.ncol
        coords = _load_matrix(self.coords_file, nrow, self.ncol)
        return coords
       
    def __del__(self):
        self.coords_file.close()

   
class ReadConfigurationsQTN(object):
    """ Read a pair of files specifying the positions and orientations of a system of rigid bodies, using quartenions
        to describe the orientations.
        Multiple configurations may be contained within the files. Each configuration consists of 3*nmol centre of mass
        coordinates in the CoM file (3 coordinates to a line) and 4*nmol orientation quartenion components in the 
        quaternions file (4 coordinates to a line).
        The coordinates may be accessed by iterating over the configuration_iterator function, which reads in each
        configuration as it is required - this is particularly efficient when you only need to read part of the file.
        The returned coordinates will be given as flat np arrays containing the 3*nmol centre-of-mass coordinates
        followed by 3*nmol angle-axis coordinates to describe the orientations of the rigid bodies.
        
        Parameters
        ----------
        positions_fname: string
            Name of a file containing centre-of-mass coordinates for the rigid bodies
        quaternion_fname: string
            Name of a file containing quartenion orientational coordinates for the rigid bodies
        nmol: int
            The number of rigid bodies in the system
    """
    
    def __init__(self, positions_fname, quaternion_fname, nmol):
        self.pos_file = open(positions_fname, "r")
        self.q_file = open(quaternion_fname, "r")
        self.nmol = nmol
        self.configuration_count = 0
    
    def _load_com_positions(self):
        """ Read the centre-of-mass coordinates from the relevant file """
        return _load_matrix(self.pos_file, self.nmol, 3)

    def _load_quaternions(self):
        """ Read quaternion coordinates from the relevant file """
        return _load_matrix(self.q_file, self.nmol, 4)
    
    def _quaternions_to_aa(self, qlist):
        """ Convert a list of quaternions to angle axis vectors """
        aa_list = map(q2aa, qlist)
        return np.array(aa_list).reshape([self.nmol,3])
    
    def load_configuration(self):
        """ Get the next complete configuration (CoM + AA coordinates) """
        coords = np.zeros([2*self.nmol, 3])
        coords[:self.nmol, :] = self._load_com_positions()
        q = self._load_quaternions()
        coords[self.nmol:, :] = self._quaternions_to_aa(q)
        return coords
        
    def configuration_iterator(self):
        """ Return the next configuration from the file every time this function is called."""
        while self.pos_file and self.q_file:
            try:
                x = self.load_configuration()
            except FileEmptyError:
                # this means we have reached the end of the file normally
                break
            self.configuration_count += 1
            yield x

    def __del__(self):
        self.pos_file.close()
        self.q_file.close()


class ReadConfigurationsCart(ReadConfigurations):
    """ Read a file containing the cartesian coordinates for a system of atoms or molecules.
        Multiple configurations may be contained within the file. Each configuration is composed of 3*nmol cartesian
        coordinates, which may represent atom positions (in which case "nmol" is the number of atoms) or centre-of-mass 
        positions. 1-column or 3-column formats are both supported.
        The coordinates may be accessed by iterating over self.configuration_iterator(), which reads in each
        configuration as it is required - this is particularly efficient when you only need to read part of the file.
        The returned coordinates will be given as np arrays with the same shape and structure as the original file.
        
        Parameters
        ----------
        coords_fname: string
            Name of the file containing centre-of-mass coordinates for the rigid bodies, followed by angle-axis
        nmol: int
            The number of rigid bodies in the system, if centre-of-mass coordinates are being read. Otherwise, this
            parameter contains the number of atoms in the system. 
        ncol: int, default 1
            The number of columns used for the coordinate matrices in the input file
    """
                 
    def load_configuration(self):
        """ Read the next configuration in from the file """
        ndof = 3*self.nmol
        assert ndof % self.ncol == 0
        nrow = ndof / self.ncol
        coords = _load_matrix(self.coords_file, nrow, self.ncol)
        return coords
    
class ReadConfigurationsList(ReadConfigurations):
    """ Read configurations from a list of input files.
        For each file, configurations will be read according to the ReadConfigurations class.
        The coordinates may be accessed by iterating over self.configuration_iterator(), which reads in each
        configuration as it is required. When one file is completed, it moves onto the next.
        
        Parameters
        ----------
        fname_list: list of strings
            A list of the input file names, in the order in which they should be read.
        nmol: int
            The number of rigid bodies in the system
        ncol: int, default 1
            The number of columns used for the coordinate matrices in the input files (must all be the same)
    """
    
    def __init__(self, fname_list, nmol, ncol=1):
        self.nmol = nmol
        self.configuration_count = 0
        self.ncol = ncol
        self.fname_list = fname_list
        
    def configuration_iterator(self):
        """ Return the next configuration from the file every time this function is called. When each file is
            finished, move on to the next until the list is exhausted. """
        for name in self.fname_list:
            self.coords_file = open(name, "r")
            while self.coords_file:
                try:
                    x = self.load_configuration()
                except FileEmptyError:
                # this means we have reached the end of the file normally
                    break
                self.configuration_count += 1
                yield x
            self.coords_file.close()        

class ReadConfigurationsCartList(ReadConfigurationsList):
    """ Read configurations in cartesian coordinates from a list of input files.
        For each file, configurations will be read according to the ReadConfigurationsCart class.
        The coordinates may be accessed by iterating over self.configuration_iterator(), which reads in each
        configuration as it is required. When one file is completed, it moves onto the next.
        
        Parameters
        ----------
        fname_list: list of strings
            A list of the input file names, in the order in which they should be read.
        nmol: int
            The number of rigid bodies in the system
        ncol: int, default 1
            The number of columns used for the coordinate matrices in the input files (must all be the same)
    """
    def load_configuration(self):
        """ Read the next configuration in from the file """
        ndof = 3*self.nmol
        assert ndof % self.ncol == 0
        nrow = ndof / self.ncol
        coords = _load_matrix(self.coords_file, nrow, self.ncol)
        return coords        
