class cage_break(object):
    """ A data structure to record the important information about a cage-breaking transition. 
    
    Parameters
    ----------
    index: integer
        The index number of the particle undergoing the cage break
    old_pos: array of floats
        numpy array containing the (CoM) coordinates of the central particle in the first configuration
    new_pos: array of floats
        numpy array containing the (CoM) coordinates of the central particle in the second configuration
    changed_neighbours: list (of lists) of integers
        A list containing the indices of all the neighbours which changed during the cage break. 
        Gained neighbours are shown by positive indices, lost neighbours by negative.
        For molecular cage breaks, there will be a list of changed neighbours for each atom in the molecule
    time: integer
        An identifier for the transition in which the cage break occurs. For a trajectory, this will be the
        timestep. For a database, this will be the number of the transition state.
    l: integer
        The length of the reversal chain up to and including this cb. l=0 means no reversals have taken place
        so far (although the cb represented by this object may later be reversed). l=1 means 1 reversal, etc.
    """
    def __init__(self, index, old_pos, new_pos, changed_neighbours, time, l=0):
        self.index = index
        self.old_pos = old_pos
        self.new_pos = new_pos
        self.changed_neighbours = changed_neighbours
        self.time = time
        self.l = l
        
    def copy(self):
        return cage_break(self.index, self.old_pos, self.new_pos, 
                          self.changed_neighbours, self.time, l=self.l)
    
    def lineprint(self):
        """ Produce a string containing the important information about the cage break in a standard format, 
        for use by other parts of the program (mostly output printing). """
        return (str(self.time)+' '+str(self.index)+' '+str(self.l)+' '+
            str(self.old_pos)+' '+str(self.new_pos)+'\n'+
                    ' '.join(map(str, self.changed_neighbours))+'\n')   


class neighchange_dcut_cb_rule(object):
    """ A cbrule object to identify from a pair of neighbour lists and configurations whether a cage break has taken place.
    In this rule, we require cb_thresh neighbours to change by at least dcut in order to register a cage break.
    This is the original distance method of de Souza and Wales, JCP 129 (2008).
    There are two very similar methods. Method 1 registers a CB when cbthresh neighbours are gained OR two are lost,
    method 2 requires a total of cbthresh neighbours to change (adding losses and gains together).
    The default is method 1.
    
    Parameters
    ----------
    dist: function
        Metric function to calculate the distance between atoms
    cb_thresh: integer
        The number of neighbours which must be gained/lost in order for a cage break to occur
    dcut: float
        The distance by which an atom must move to count as being lost/gained (in addition to moving across the
        boundary of the nearest-neighbour shell)
    reversal_manager: reversal_manager object, default None
        An object to determine whether a newly-identified cage-break is in fact a reversal of the previous cb.
    method: integer, default 1
        Specify which of the two methods are used to identify cage breaks (see above).
        
    Returns
    -------
    this_cb: cage_break object
        An object holding data about the current transition, if a cage break took place. If no cage break took
        place (or if a reversal chain is ongoing - see reversal_manager) then this object returns None.
    """ 
    def __init__(self, dist, cb_thresh, dcut, reversal_manager=None, method=1):
        self.dist = dist
        self.reversal_manager = reversal_manager
        
        self.cb_thresh = cb_thresh
        self.dcut = dcut

        self.count=0

        self.method = method # Selects which of two very similar definitions to use
    
    def __call__(self, index, x0, x1, old_neighbours, neighbours, time):
        """ Identify whether a cage break takes place.
        
        Parameters
        ----------
        index: integer
            The index of the atom which is being tested for a cage break
        x0, x1: arrays of floats
            Coordinate arrays for the entire system in the two configurations being compared
        old_neighbours: list of integers
            Atom indices for the neighbours of atom "index" in the old configuration
        neighbours: list of integers
            Atom indices for the neighbours of atom "index" in the new configuration
        time: integer
            A label for the current pair of configurations being tested (when examining a trajectory, this
            indexes the current timestep). This is only used for bookkeeping purposes.
        
        Attributes
        ----------
        changed_neighbours: list
            A list of indices for atoms which have been lost (negative indices) or gained (positive) as neighbours
            between configuration x0 and x1.
            
        Returns
        -------
        this_cb: cage_break object
            An object holding data about the current transition, if a cage break took place. If no cage break took
            place (or if a reversal chain is ongoing - see reversal_manager) then this object returns None.                        
        """
        # This counts the number of changed neighbours (lost, then gained)
        cb_count = [0,0]
        # Store a list of changed neighbours
        changed_neighbours = []
        # First see whether any of the old neighbours have moved out of the neighbour shell
        for i in old_neighbours:
            if i in neighbours:
                # The old neighbour is still a neighbour.
                continue
            else:
                # The old neighbour is missing. But did it move far enough to be considered a 
                # cage break?
                new_d = self.dist(x1[index], x1[i]) # New distance between the neighbour and the central atom
                old_d = self.dist(x0[index], x0[i]) # Old distance between the neighbour and the central atom

                # Test to see whether the atom has moved far enough to change neighbour shells
                if (abs(new_d-old_d) > self.dcut):
                    # Record it as a lost neighbour.
                    cb_count[0] += 1
                    changed_neighbours.append(-i)
                    
        # Are there any new atoms in the neighbour shell?
        for i in neighbours:
            if i in old_neighbours:
                continue
            else:
                # A new atom in the shell. Did it move far enough to be a cage break?
                new_d = self.dist(x1[index], x1[i])                
                old_d = self.dist(x0[index], x0[i])                

                # Test to see whether the atom has moved far enough to change neighbour shells
                if (abs(old_d - new_d) > self.dcut):
                    cb_count[1] += 1
                    changed_neighbours.append(i)                  

        # Have enough atoms moved for this step to count as a cage break?
        if ((self.method==1 and (cb_count[0] >= self.cb_thresh or cb_count[1] >=self.cb_thresh))
            or (self.method==2 and (cb_count[0]+cb_count[1] >= self.cb_thresh))):
            
            # The following line may give a lot of output, so has been commented out.
#            print "Atom ", index, "has undergone a CB"
                       
            # Cage break detected: save data
            this_cb = cage_break(index, x0[index], x1[index], changed_neighbours, time)
            
            # If there is a reversal manager, pass the cb in to it, and find out whether the current cb is
            # just a reversal of the previous cb or not. The reversal manager decides whether to pass out the
            # cb to the rest of the program.
            if self.reversal_manager is None:
                return this_cb
            else:
                return self.reversal_manager(this_cb)
        else: # No cb detected: return.
            return None

class binary_neighchange_dcut_cbrule(neighchange_dcut_cb_rule):
    """ A variation on neighchange_dcut_cbrule for binary AB systems such as BLJ or Silica.
    A different movement cutoff is required for each type of atom pair: AA, AB and BB.
    This is assumed to be for atomic systems only, but it could probably be adapted to handle molecules.
    
    Parameters
    ----------
    dist: function
        Metric function to calculate the distance between atoms
    cb_threshA: integer
        The number of neighbours which must be gained/lost in order for an A-type atom to undergo a cb.
    dcutAA: float
        Movement cutoff for an A atom moving towards or away from another A atom
    dcutAB: float
        Movement cutoff for an B atom moving towards or away from an A atom (or vice versa)
    dcutBB: float
        Movement cutoff for an B atom moving towards or away from another B atom (or vice versa)
    nmol: integer
        Number of atoms in the system (type A + type B)
    ntypeA: integer
        Number of type-A atoms in the system
    reversal_manager: reversal_manager object, default None
        An object to determine whether a newly-identified cage-break is in fact a reversal of the previous cb.
    method: integer, default 1
        Specify which of the two methods are used to identify cage breaks (see above).
    cb_threshB: float, default 0
        The number of neighbours which must be gained/lost in order for an A-type atom to undergo a cb.
	If this is not specified manually, it will be set equal to cb_threshA.
        
    Returns
    -------
    this_cb: cage_break object
        An object holding data about the current transition, if a cage break took place. If no cage break took
        place (or if a reversal chain is ongoing - see reversal_manager) then this object returns None.   
    """

    def __init__(self, dist, cb_threshA, dcutAA, dcutAB, dcutBB, nmol, ntypeA, reversal_manager=None, method=1, cb_threshB=0):
        self.dist = dist
        self.reversal_manager = reversal_manager
        
        self.type = [0]*ntypeA+[1]*(nmol-ntypeA)
        self.dcuts = [dcutAA, dcutAB, dcutBB]

	if cb_threshB:
	    self.cbthresh = [cb_threshA, cb_threshB]
	else:
            # Then the threshold is assumed to be the same for both types of atom
	    self.cbthresh = [cb_threshA, cb_threshA]

        self.count=0

        self.method = method # Selects which of two very similar definitions to use
        
    
    def __call__(self, index, x0, x1, old_neighbours, neighbours, time):
        # This counts the number of changed neighbours (lost, then gained)
        cb_count = [0,0]
        # Store a list of changed neighbours
        changed_neighbours = []
        # First see whether any of the old neighbours have moved out of the neighbour shell
        for i in old_neighbours:
            if i in neighbours:
                # The old neighbour is still a neighbour.
                continue
            else:
                # The old neighbour is missing. But did it move far enough to be considered a 
                # cage break?
                new_d = self.dist(x1[index], x1[i]) # New distance between the neighbour and the central atom
                old_d = self.dist(x0[index], x0[i]) # Old distance between the neighbour and the central atom

                # Test to see whether the atom has moved far enough to change neighbour shells
                if (abs(new_d-old_d) > self.dcuts[self.type[i]+self.type[index]]):
                    # Record it as a lost neighbour.
                    cb_count[0] += 1
                    changed_neighbours.append(-i)
                    
        # Are there any new atoms in the neighbour shell?
        for i in neighbours:
            if i in old_neighbours:
                continue
            else:
                # A new atom in the shell. Did it move far enough to be a cage break?
                new_d = self.dist(x1[index], x1[i])                
                old_d = self.dist(x0[index], x0[i])                

                # Test to see whether the atom has moved far enough to change neighbour shells
                if (abs(old_d - new_d) > self.dcuts[self.type[i]+self.type[index]]):
                    cb_count[1] += 1
                    changed_neighbours.append(i)                  

        # Have enough atoms moved for this step to count as a cage break?
        if ((self.method==1 and (cb_count[0] >= self.cbthresh[self.type[index]] or cb_count[1] >=self.cbthresh[self.type[index]]))
            or (self.method==2 and (cb_count[0]+cb_count[1] >= self.cbthresh[self.type[index]]))):
            
            # The following line may give a lot of output, so has been commented out.
#            print "Atom ", index, "has undergone a CB"
                       
            # Cage break detected: save data
            this_cb = cage_break(index, x0[index], x1[index], changed_neighbours, time)
            
            # If there is a reversal manager, pass the cb in to it, and find out whether the current cb is
            # just a reversal of the previous cb or not. The reversal manager decides whether to pass out the
            # cb to the rest of the program.
            if self.reversal_manager is None:
                return this_cb
            else:
                return self.reversal_manager(this_cb)
        else: # No cb detected: return.
            return None
